#!/bin/ash

red='\033[0;31m'
green='\033[0;32m'
yellow='\033[0;33m'
nc='\033[0m' # No Color

files_install_path="/root/app"
log_path="/var/log/rutapp.log"
usb_log_glob_path="/mnt/sd*1/rut-logs/rutapp.log"
log_directory=$(dirname "${log_path}")
logrotate_dir="/etc/logrotate.d"
logrotate_file="${logrotate_dir}/rutapp"
usb_logrotate_file="${logrotate_dir}/rutapp-usb"

libraries_to_install="libcrypto.so.3 libssl.so.3 libstdc++.so.6.0.25"
lib_search_paths="/lib /usr/lib /usr/local/lib"
lib_install_path="/usr/local/lib"
lib_web_url="https://svannet.com/downloads/packages/libs"
force_installation=false
reset_settings=false
number_of_instruments=""

terminate_application() {
  local rutapp_pid=$(ps | grep -i 'app/[r]utapp' | awk '{print $1;}')
  if [ -n "$rutapp_pid" ]; then
    echo -e "${yellow}Application is running, terminating...${nc}"
    kill -9 $rutapp_pid > /dev/null 2>&1
  fi
}

create_log_directory() {
  if ! [ -d "${log_directory}" ]; then
    mkdir -p "${log_directory}"
  fi
}

create_logrotate_file() {
  echo "Creating logrotate files..."
  cat > "${logrotate_file}" <<EOF
${log_path} {
    daily
    rotate 4
    maxsize 300k
    compress
    missingok
    notifempty
    copytruncate
}
EOF

cat > "${usb_logrotate_file}" <<EOF
${usb_log_glob_path} {
    su root root
    daily
    rotate 14
    compress
    missingok
    notifempty
    copytruncate
}
EOF
}

chmod_script_files() {
  local script_dir="$1"
  chmod +x "${script_dir}"/*.sh
}

install_application_files() {
  local script_dir="$1"

  chmod_script_files "$script_dir"
  echo "Copying application files..."
  if ! [ -d "${files_install_path}" ]; then
    mkdir -p "${files_install_path}"
  fi

  terminate_application
  # Copy the binary files
  cp -r "${script_dir}/rutapp" "${files_install_path}/"
  cp -r "${script_dir}/openRTSP" "${files_install_path}/"

  # Copy timezone file
  cp -r "${script_dir}/timezones.ini" "${files_install_path}/"

  # Copy the .sh files
  cp -r "${script_dir}"/*.sh "${files_install_path}/"
  # Copy the cert files
  cp -r "${script_dir}/cert" "${files_install_path}/"

  if [ ! -d "${files_install_path}/logs" ]; then
    mkdir -p "${files_install_path}/logs"
  fi

  # Check differences for ${script_dir}/settings.ini, if there are any ask to overwrite
  if [ -f "${script_dir}/settings.ini" ]; then
    if [ -f "${files_install_path}/settings.ini" ]; then
      if ! cmp "${script_dir}/settings.ini" "${files_install_path}/settings.ini" > /dev/null 2>&1; then
        if $reset_settings; then
          echo -e "${yellow}settings.ini already exists and is different from the one in the package. Overwriting...${nc}"
          cp -r "${script_dir}/settings.ini" "${files_install_path}/"
        else
          echo -e "${yellow}settings.ini already exists and is different from the one in the package. If you wish to replace it, use the --reset-settings option${nc}"
        fi
      fi
    else
      cp -r "${script_dir}/settings.ini" "${files_install_path}/"
    fi
  fi

    # Check differences for ${script_dir}/numberOfDevices, if there are any ask to overwrite
    if [ -f "${script_dir}/numberOfDevices" ]; then
      if [ -f "${files_install_path}/numberOfDevices" ]; then
        if ! cmp "${script_dir}/numberOfDevices" "${files_install_path}/numberOfDevices" > /dev/null 2>&1; then
          if $reset_settings; then
            echo -e "${yellow}numberOfDevices already exists and is different from the one in the package. Overwriting...${nc}"
            cp -r "${script_dir}/numberOfDevices" "${files_install_path}/"
          else
            echo -e "${yellow}numberOfDevices already exists and is different from the one in the package. If you wish to replace it, use the --reset-settings option${nc}"
          fi
        fi
      else
        cp -r "${script_dir}/numberOfDevices" "${files_install_path}/"
      fi
    fi

}

is_non_lib_package() {
  # Check if libs is present, if not this is a non-lib package.
  if [ -d "./libs" ]; then
    return 1 # false
  else
    return 0 # true
  fi
}

does_library_exist() {
  local file="$1"
  for path in $lib_search_paths; do
    if [ -e "${path}/${file}" ]; then
      return 0 # true
    else
      local similar_files=$(find "${path}" -maxdepth 1 -iname "${file}.*")
      if [ -n "$similar_files" ]; then
        return 0 # true
      fi
    fi
  done

  return 1 # false
}

install_libraries_from_web() {
  for library_file in $libraries_to_install; do
    if ! does_library_exist "$library_file"; then
      echo "Installing library: ${library_file}"
      if ! wget -O "${lib_install_path}/${library_file}" "${lib_web_url}/${library_file}"; then
        echo -e "${red}Failed to download library: ${library_file}${nc}"
        exit 1
      fi
      echo " Library installed: ${library_file}"
    else
      echo " Library already installed: ${library_file}"
    fi
  done
}

install_libraries_from_package() {
  for library_file in $libraries_to_install; do
    if ! [ -f "./libs/${library_file}" ]; then
      echo -e "${red}Library file not found: ${library_file}${nc}"
      exit 1
    fi

    if does_library_exist "$library_file"; then
      echo " Library already installed: ${library_file}"
      rm "./libs/$file"
    else
      echo "Installing library: ${library_file}"
      mv "./libs/$file" "${lib_install_path}"
      echo " Library installed: ${library_file}"
    fi
  done
}

install_libraries() {
  echo "Installing libraries..."
  if is_non_lib_package; then
    install_libraries_from_web
    return
  fi

  install_libraries_from_package
}

check_and_update_crontab_entry() {
  local command_to_check="$1"
  local new_entry="$2"
  local old_entry=$(crontab -l | grep "$command_to_check")

  if [ -n "$old_entry" ]; then
    remove_cron_entry "$old_entry"
  fi

  add_cron_entry "$new_entry"
}

remove_cron_entry() {
  local entry_to_remove="$1"
  crontab -l | grep -vF "$entry_to_remove" | crontab -
}

add_cron_entry() {
  local new_entry="$1"
  (crontab -l; echo "$new_entry") | crontab -
}

# Install crontab entries
check_and_update_crontab_entries() {
  echo "Installing crontab entries..."
  check_and_update_crontab_entry "${files_install_path}/restarter.sh" "* * * * * ${files_install_path}/restarter.sh >> ${log_path}"
  check_and_update_crontab_entry "${files_install_path}/downloadCA.sh" "0 * * * * ${files_install_path}/downloadCA.sh >> ${files_install_path}/logs/downloadCA.log"
  check_and_update_crontab_entry "${files_install_path}/device-watchdog.sh" "*/15 * * * * ${files_install_path}/device-watchdog.sh >> ${files_install_path}/logs/watchdog.log 2>&1"

  local usb_logrotate_old_entry=$(crontab -l | grep "logrotate /etc/logrotate.d")
  if [ -n "$usb_logrotate_old_entry" ]; then
    remove_cron_entry "$usb_logrotate_old_entry"
  fi
  check_and_update_crontab_entry "logrotate -s /etc/logrotate.status" "*/10 * * * * logrotate -s /etc/logrotate.status ${logrotate_dir}"
}

associate_usb_devices_with_serial() {
  echo "Associating USB devices with serial..."
  associate_usb_device_with_serial "0017 0001"
  associate_usb_device_with_serial "0017 0002"
}

# Install driver for USB device
associate_usb_device_with_serial() {
  local device_id="$1"
  check_and_update_usb_serial_new_id "$device_id"
  add_driver_association_to_rc_local "$device_id"
}

check_and_update_usb_serial_new_id() {
  local entry_to_check="$1"

  if ! grep -qF "$entry_to_check" /sys/bus/usb-serial/drivers/generic/new_id; then
    echo "$entry_to_check" >> /sys/bus/usb-serial/drivers/generic/new_id
  fi
}

add_driver_association_to_rc_local() {
  local device_id="$1"
  if ! grep -qF "echo $device_id" /etc/rc.local; then
    sed -i -e "\$i \echo $device_id > /sys/bus/usb-serial/drivers/generic/new_id" /etc/rc.local
  fi
}

add_number_of_devices_reset_to_rc_local() {
  if ! grep -qF "echo 0 > /root/app/numberOfDevices" /etc/rc.local; then
    sed -i -e "\$i \echo 0 > /root/app/numberOfDevices" /etc/rc.local
  fi
}

does_logrotate_exist() {
  if ! [ -x "$(command -v logrotate)" ]; then
    return 1 # false
  fi

  return 0 # true
}

install_logrotate() {
  if does_logrotate_exist; then
    return;
  fi

  echo "Installing logrotate..."

  # It's not available in the device custom repository, download and install locally
  wget -O /tmp/logrotate_3.17.0-1_mipsel_24kc.ipk https://downloads.openwrt.org/releases/21.02.0/packages/mipsel_24kc/packages/logrotate_3.17.0-1_mipsel_24kc.ipk
  wget -O /tmp/libpopt0_1.16-2_mipsel_24kc.ipk https://downloads.openwrt.org/releases/21.02.0/packages/mipsel_24kc/base/libpopt0_1.16-2_mipsel_24kc.ipk
  opkg install /tmp/libpopt0_1.16-2_mipsel_24kc.ipk /tmp/logrotate_3.17.0-1_mipsel_24kc.ipk
  rm /tmp/logrotate_3.17.0-1_mipsel_24kc.ipk /tmp/libpopt0_1.16-2_mipsel_24kc.ipk
}

update_number_of_instruments() {
  if [ -n "$number_of_instruments" ]; then
    echo "$number_of_instruments" > "${files_install_path}/numberOfDevices"
  fi
}

print_usage() {
    echo "Usage: $0 [options]"
    echo "Options:"
    echo "  -h, --help              Display this help message"
    echo "  -l, --libs-url          Set the library web URL"
    echo "  -f, --force             Force installation even if the application directory already exists"
    echo "  --reset-settings        Reset the settings.ini file to default values (if applicable)."
    echo "  --number-of-instruments Set the number of instruments for the watchdog script"
}

while [ $# -gt 0 ]; do
  case $1 in
    --help | -h)
      print_usage
      exit 0
      ;;
    --libs-url | -l)
      shift
      if [ -z "$1" ] || [ "${1#-}" != "$1" ]; then
        echo "ERROR: No library web URL provided"
        exit 1
      fi
      lib_web_url="$1"
      ;;
    --force | -f)
      force_installation=true
      ;;
    --reset-settings)
      reset_settings=true
      ;;
    --number-of-instruments)
      shift
      if [ -z "$1" ] || [ "${1#-}" != "$1" ]; then
        echo "ERROR: No number of instruments provided"
        exit 1
      fi
      number_of_instruments="$1"
      ;;
    -*)
      echo "ERROR: Unrecognized option: $1"
      exit 1
      ;;
    *)
      ;;
  esac
  shift
done

lib_web_url="${lib_web_url%/}"

if [ "$(id -u)" -ne 0 ]; then
  echo -e "${red}Error: This script must be run as root${nc}"
  exit 1
fi

echo "Installing the rutapp application..."
if [ -d "${files_install_path}" ]; then
  if ! $force_installation; then
    echo -e "${yellow}Application directory already exists. If you wish to replace the files without resetting the settings, use the --force option${nc}"
    exit 1
  else
    echo -e "${yellow}Application directory already exists. Overwriting...${nc}"
    if ! $reset_settings; then
      echo -e "${yellow}The settings won't be reset. If you want to reset the settings, use the --reset-settings option${nc}"
    else
      echo -e "${yellow}The settings will be reset${nc}"
    fi
  fi
fi

install_libraries
install_logrotate
create_log_directory
create_logrotate_file
install_application_files "$(dirname "$(readlink -f "$0")")"
update_number_of_instruments
associate_usb_devices_with_serial
add_number_of_devices_reset_to_rc_local
check_and_update_crontab_entries

echo -e "${green}Application installed successfully${nc}"
